/*:
 * @target MZ
 * @plugindesc [Yui] Sanfte, warme Glühwürmchen (Feuerflies) als Partikel-Overlay auf Maps (ohne Assets).
 * @author Yui
 * @url -
 *
 * @help
 * Fügt der Map ein sanftes Glühwürmchen-Overlay hinzu:
 * - Weiches Leuchten (Additive Blend), leichtes Pulsieren
 * - Zufälliges Driften, sanftes Winden
 * - Anpassbar per Parameter und Plugin-Commands
 *
 * TIPP:
 * - Layer „Above Characters“ lässt die Partikel über den Figuren schweben.
 * - Stelle die Dichte mapweise per Plugin-Command ein (z.B. nachts höher).
 *
 * Plugin Commands:
 *   YuiFireflies Enable
 *   YuiFireflies Disable
 *   YuiFireflies SetDensity <Zahl>
 *
 * @command Enable
 * @text Enable
 * @desc Schaltet die Glühwürmchen ein.
 *
 * @command Disable
 * @text Disable
 * @desc Schaltet die Glühwürmchen aus.
 *
 * @command SetDensity
 * @text Set Density
 * @desc Setzt die Zielanzahl der Glühwürmchen (0 = aus).
 * @arg count
 * @type number
 * @min 0
 * @default 50
 * @text Count
 *
 * @param EnabledByDefault
 * @text Enabled (Default)
 * @type boolean
 * @default true
 *
 * @param Layer
 * @text Layer
 * @type select
 * @option Below Characters
 * @value below
 * @option Above Characters
 * @value above
 * @default above
 *
 * @param Density
 * @text Density (Count)
 * @type number
 * @min 0
 * @default 50
 *
 * @param Color
 * @text Glow Color (hex)
 * @type string
 * @default #ffdd88
 *
 * @param MinSize
 * @text Min Size (px)
 * @type number
 * @min 2
 * @default 3
 *
 * @param MaxSize
 * @text Max Size (px)
 * @type number
 * @min 2
 * @default 6
 *
 * @param MinSpeed
 * @text Min Speed (px/frame)
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.15
 *
 * @param MaxSpeed
 * @text Max Speed (px/frame)
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.45
 *
 * @param DriftStrength
 * @text Drift Strength
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.02
 *
 * @param PulseSpeed
 * @text Pulse Speed
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.02
 *
 * @param BaseOpacity
 * @text Base Opacity (0-255)
 * @type number
 * @min 0
 * @max 255
 * @default 210
 *
 * @param ZIndex
 * @text Z-Index (Layer sort)
 * @type number
 * @min 0
 * @default 3
 *
 */

(() => {
  const PLUGIN_NAME = "Yui_Fireflies";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const P = {
    enabled: params.EnabledByDefault === "true",
    layer: String(params.Layer || "above"),
    density: Number(params.Density || 50),
    color: String(params.Color || "#ffdd88"),
    minSize: Number(params.MinSize || 3),
    maxSize: Number(params.MaxSize || 6),
    minSpeed: Number(params.MinSpeed || 0.15),
    maxSpeed: Number(params.MaxSpeed || 0.45),
    drift: Number(params.DriftStrength || 0.02),
    pulse: Number(params.PulseSpeed || 0.02),
    baseOpacity: Number(params.BaseOpacity || 210),
    z: Number(params.ZIndex || 3),
  };

  // ---------- Plugin Commands ----------
  PluginManager.registerCommand(PLUGIN_NAME, "Enable", () => {
    $gameSystem._yuiFirefliesEnabled = true;
  });
  PluginManager.registerCommand(PLUGIN_NAME, "Disable", () => {
    $gameSystem._yuiFirefliesEnabled = false;
  });
  PluginManager.registerCommand(PLUGIN_NAME, "SetDensity", args => {
    const n = Number(args.count || 0);
    $gameSystem._yuiFirefliesDensity = Math.max(0, n);
  });

  // Defaults on new game
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function () {
    _Game_System_initialize.call(this);
    if (this._yuiFirefliesEnabled == null) this._yuiFirefliesEnabled = P.enabled;
    if (this._yuiFirefliesDensity == null) this._yuiFirefliesDensity = P.density;
  };

  // ---------- Firefly System ----------
  class YuiFirefly extends Sprite {
    constructor(tex, w, h, config) {
      super(new PIXI.Texture(tex));
      this._config = config;
      this.anchor.set(0.5);
      this.reset(true);
      this.blendMode = PIXI.BLEND_MODES.ADD;
      this.zIndex = config.z;
    }

    reset(randomPos = false) {
      const c = this._config;
      const s = c.minSize + Math.random() * (c.maxSize - c.minSize);
      this.scale.set(s / 32); // Basisgröße der Textur ≈ 32px
      this._speed = c.minSpeed + Math.random() * (c.maxSpeed - c.minSpeed);
      this._dir = Math.random() * Math.PI * 2;
      this._drift = (Math.random() * 2 - 1) * c.drift;
      this._pulsePhase = Math.random() * Math.PI * 2;
      this._pulseSpeed = c.pulse * (0.7 + Math.random() * 0.6);

      // Startposition
      if (randomPos) {
        this.x = Math.random() * Graphics.width;
        this.y = Math.random() * Graphics.height;
      } else {
        // Spawn leicht am Rand
        const side = Math.floor(Math.random() * 4);
        if (side === 0) { this.x = -10; this.y = Math.random() * Graphics.height; }
        if (side === 1) { this.x = Graphics.width + 10; this.y = Math.random() * Graphics.height; }
        if (side === 2) { this.x = Math.random() * Graphics.width; this.y = -10; }
        if (side === 3) { this.x = Math.random() * Graphics.width; this.y = Graphics.height + 10; }
      }

      this.alpha = (P.baseOpacity / 255) * 0.6;
    }

    update() {
      super.update();
      // sanfter Drift der Richtung
      this._dir += this._drift * (Math.random() * 0.5 + 0.5);
      const vx = Math.cos(this._dir) * this._speed;
      const vy = Math.sin(this._dir) * this._speed;
      this.x += vx;
      this.y += vy;

      // Pulsierendes Leuchten
      this._pulsePhase += this._pulseSpeed;
      const pulse = (Math.sin(this._pulsePhase) + 1) / 2; // 0..1
      this.alpha = ((P.baseOpacity / 255) * 0.4) + pulse * ((P.baseOpacity / 255) * 0.6);

      // sanft „zurück warp’en“, falls zu weit weg
      const margin = 32;
      if (this.x < -margin || this.x > Graphics.width + margin || this.y < -margin || this.y > Graphics.height + margin) {
        this.reset(false);
      }
    }
  }

  class YuiFirefliesLayer extends PIXI.Container {
    constructor() {
      super();
      this.zIndex = P.z;
      this._enabled = false;
      this._targetCount = 0;
      this._sprites = [];
      this._tex = this._makeGlowTexture(P.color);
    }

    setupFromSystem() {
      this.setEnabled($gameSystem._yuiFirefliesEnabled);
      this.setDensity($gameSystem._yuiFirefliesDensity);
    }

    setEnabled(b) {
      this._enabled = !!b;
      this.visible = this._enabled;
    }

    setDensity(n) {
      this._targetCount = Math.max(0, n|0);
      this._rebalance();
    }

    _rebalance() {
      // Entfernen
      while (this._sprites.length > this._targetCount) {
        const s = this._sprites.pop();
        this.removeChild(s);
      }
      // Hinzufügen
      while (this._enabled && this._sprites.length < this._targetCount) {
        const firefly = new YuiFirefly(this._tex, Graphics.width, Graphics.height, P);
        this.addChild(firefly);
        this._sprites.push(firefly);
      }
    }

    update() {
      if (!this._enabled) return;
      for (const s of this._sprites) s.update();
    }

    _makeGlowTexture(hex) {
      // Erzeuge eine weiche runde Glow-Textur via PIXI.Graphics -> Texture
      const g = new PIXI.Graphics();
      const col = PIXI.utils.string2hex(hex || "#ffdd88");
      const R = 16; // Basisradius
      for (let r = R; r > 0; r--) {
        const alpha = (r / R) * (r / R); // weich nach innen heller
        g.beginFill(col, alpha);
        g.drawCircle(R, R, r);
        g.endFill();
      }
      const tex = PIXI.RenderTexture.create({ width: R*2, height: R*2 });
      // Achtung: use renderer from SceneManager
      SceneManager._scene && SceneManager._scene._spriteset && SceneManager._scene._spriteset.children;
      // Fallback auf globalRenderer
      const renderer = Graphics.app.renderer;
      renderer.render(g, { renderTexture: tex });
      g.destroy(true);
      return tex;
    }
  }

  // ---------- Hook in Spriteset_Map ----------
  const _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
  Spriteset_Map.prototype.createLowerLayer = function () {
    _Spriteset_Map_createLowerLayer.call(this);
    this._yuiFireflies = new YuiFirefliesLayer();
    this._yuiFireflies.zIndex = P.z;

    if (P.layer === "below") {
      this._tilemap.addChild(this._yuiFireflies);
    } else {
      // über den Charakteren
      this._upperLayer.addChild(this._yuiFireflies);
    }
    this._yuiFireflies.setupFromSystem();
    this.sortChildren();
  };

  const _Spriteset_Map_update = Spriteset_Map.prototype.update;
  Spriteset_Map.prototype.update = function () {
    _Spriteset_Map_update.call(this);
    if (this._yuiFireflies) {
      // Falls sich System-Flags live ändern (durch Commands), Dichte anpassen
      if (this._yuiFireflies.visible !== $gameSystem._yuiFirefliesEnabled) {
        this._yuiFireflies.setEnabled($gameSystem._yuiFirefliesEnabled);
      }
      if (this._yuiFireflies._targetCount !== $gameSystem._yuiFirefliesDensity) {
        this._yuiFireflies.setDensity($gameSystem._yuiFirefliesDensity);
      }
      this._yuiFireflies.update();
    }
  };

})();
